import argparse, json
from pathlib import Path
import numpy as np

def read_D_map(path):
    data = json.loads(Path(path).read_text(encoding="utf-8"))
    if isinstance(data, dict):
        items = sorted([(float(k), float(v)) for k,v in data.items()], key=lambda t: t[0])
    elif isinstance(data, list):
        items = sorted([(float(e["n"]), float(e["D"])) for e in data], key=lambda t: t[0])
    else:
        raise ValueError("Unsupported D map format")
    return dict(items)

def spectral_radius(M: np.ndarray) -> float:
    vals = np.linalg.eigvals(M)
    return float(np.max(np.abs(vals)))

def main():
    ap = argparse.ArgumentParser(description="Compute leading eigenvalues per-context and compare to D(n).")
    ap.add_argument("--kernels", default="results", help="Folder with kernels/*.npy and kernel_index.json")
    ap.add_argument("--D", required=True, help="Path to D_of_n.json")
    ap.add_argument("--out", default="results/kernel_eigs.csv", help="Output CSV path")
    ap.add_argument("--tol", type=float, default=1e-6, help="Tolerance for |lambda_max - D(n)|")
    args = ap.parse_args()

    Dmap = read_D_map(args.D)
    idx = json.loads(Path(args.kernels, "kernel_index.json").read_text(encoding="utf-8"))
    lines = ["n,D_target,lambda_max,abs_diff,pass\n"]
    max_abs_diff = 0.0
    for ent in idx:
        n = float(ent["n"])
        M = np.load(ent["path"])
        lam = spectral_radius(M)
        Dn = float(Dmap[n])
        diff = abs(lam - Dn)
        ok = diff <= args.tol
        max_abs_diff = max(max_abs_diff, diff)
        lines.append(f"{n},{Dn},{lam},{diff},{int(ok)}\n")
    Path(args.out).parent.mkdir(parents=True, exist_ok=True)
    Path(args.out).write_text("".join(lines), encoding="utf-8")
    print(f"Wrote {args.out}. max |lambda_max - D(n)| = {max_abs_diff}")
    if max_abs_diff > args.tol:
        print(f"WARNING: max abs diff {max_abs_diff} > tol {args.tol}")

if __name__ == "__main__":
    main()
